import sys
import os
import json
import datetime
import webbrowser
import subprocess

import clr

clr.AddReference('System')
clr.AddReference('System.Windows.Forms')
clr.AddReference('System.Drawing')
clr.AddReference('System.Threading')

import System
from System.Windows.Forms import *
from System.Drawing import *
from System.Threading import *






CONFIG_FILEPATH = os.path.join(os.path.abspath(os.path.dirname(__file__)), 'config.json')






class MainForm(Form):

    def __init__(self, output_script_path):
        
        Form.__init__(self)

        self.output_script_path = output_script_path

        self.ClientSize = Size(800, 600)
        self.StartPosition = FormStartPosition.CenterScreen
        #self.MaximizeBox = False

        self.Text = 'W3LuaEnv   25.03.21'
        self.ShowIcon = False


        self.Menu = MainMenu()

        self.btn_open_output_script = MenuItem()
        self.btn_open_output_script.Text = 'output script (debug)'
        self.btn_open_output_script.Click += lambda *args: (os.startfile(self.output_script_path) if os.path.exists(self.output_script_path) else None)

        self.btn_open_project_dir = MenuItem()
        self.btn_open_project_dir.Text = 'project dir'
        self.btn_open_project_dir.Click += lambda *args: (subprocess.Popen(f'explorer.exe "{os.path.split(self.main_script_path)[0]}"') if self.main_script_path else None)

        self.btn_web = MenuItem()
        self.btn_web.Text = 'website'
        self.btn_web.Click += lambda *args: webbrowser.open_new_tab('https://w3luaenv.corexe.app')

        self.Menu.MenuItems.Add(self.btn_open_output_script)
        self.Menu.MenuItems.Add(self.btn_open_project_dir)
        self.Menu.MenuItems.Add(self.btn_web)


        PADDING = 5

        self.lbl_script = Label()
        self.lbl_script.Location = Point(PADDING, PADDING)
        self.lbl_script.AutoSize = False
        self.lbl_script.Size = Size(int(self.ClientSize.Width*0.80) - 12, self.lbl_script.Size.Height)
        self.lbl_script.BackColor = ColorTranslator.FromHtml('#1E3A5F')
        self.lbl_script.ForeColor = ColorTranslator.FromHtml('#F1F1F1')
        self.lbl_script.TextAlign = ContentAlignment.MiddleCenter
        self.lbl_script.Text = ''
        self.lbl_script.Anchor = AnchorStyles.Left | AnchorStyles.Right | AnchorStyles.Top

        self.btn_select = Button()
        self.btn_select.Text = 'select main script'
        self.btn_select.Font = Font(self.btn_select.Font, FontStyle.Bold)
        self.btn_select.Location = Point(self.lbl_script.Location.X + self.lbl_script.Size.Width + 6, PADDING)
        self.btn_select.Size = Size(int(self.ClientSize.Width*0.20) - 6, self.btn_select.Size.Height)
        self.btn_select.Anchor = AnchorStyles.Right | AnchorStyles.Top

        self.panel = Panel()
        self.panel.AutoScroll = True 
        self.panel.Location = Point(PADDING, self.lbl_script.Size.Height + PADDING*2)
        self.panel.Size = Size(self.ClientSize.Width - PADDING*2, self.ClientSize.Height - self.lbl_script.Size.Height - PADDING*3)
        self.panel.Anchor = AnchorStyles.Left | AnchorStyles.Right | AnchorStyles.Top | AnchorStyles.Bottom
        self.panel.HorizontalScroll.Visible = True
        self.panel.VerticalScroll.Visible = True
        self.panel.BackColor = ColorTranslator.FromHtml('#272822')


        self.Controls.Add(self.lbl_script)
        self.Controls.Add(self.btn_select)
        self.Controls.Add(self.panel)

        self.Load += lambda *args: setattr(self, 'MinimumSize', self.Size)

        self.btn_select.Click += self.select_main_script

        self.open_file_dialog = OpenFileDialog()

        if not os.path.exists(CONFIG_FILEPATH):

            self.config = {'init_dir': 'C:\\'}

            with open(CONFIG_FILEPATH, 'w') as f:
                f.write(json.dumps(self.config))

        else:

            with open(CONFIG_FILEPATH, 'r') as f:
                self.config = json.loads(f.read())

        self.open_file_dialog.InitialDirectory = self.config.get('init_dir')
        self.open_file_dialog.DefaultExt = 'lua'
        self.open_file_dialog.Filter = 'lua|*.lua'
        
        self.main_script_path = None
        self.wait = False
        self.last_dt = datetime.datetime.now()



    def select_main_script(self, *args):

        self.wait = True

        if self.open_file_dialog.ShowDialog() == DialogResult.OK:

            self.main_script_path = self.lbl_script.Text = self.open_file_dialog.FileName

            self.open_file_dialog.InitialDirectory = init_dir = os.path.split(self.open_file_dialog.FileName)[0]

            with open(CONFIG_FILEPATH, 'r+') as f:
                self.config = json.loads(f.read())
                self.config['init_dir'] = init_dir
                f.seek(0)
                f.write(json.dumps(self.config))
                f.truncate()

        self.wait = False


    def mk_log_lbl(self, text, color):
        lbl = Label()
        lbl.Text = text
        lbl.AutoSize = False
        lbl.ForeColor = ColorTranslator.FromHtml(color)
        lbl.Location = Point(0, (self.panel.Controls.Count * lbl.Size.Height) - self.panel.VerticalScroll.Value)
        lbl.Size = Size(self.panel.Size.Width, lbl.Size.Height)
        lbl.Anchor = AnchorStyles.Left | AnchorStyles.Right | AnchorStyles.Top
        lbl.Font = Font('Courier New', 9)
        return lbl

    def log(self, *args, **kwargs):
        self.Invoke(System.Action(lambda: self._log(*args, **kwargs)))

    def _log(self, text, color='white'):

        lines = text.split('\n')

        if len(lines) <= 0:
            return
        

        dt = datetime.datetime.now()


        if (dt - self.last_dt).total_seconds() > 1 and self.panel.Controls.Count > 0 and self.panel.Controls[self.panel.Controls.Count - 1].Text.strip() != '':
            lbl = Label()
            lbl = self.mk_log_lbl('', color)
            self.panel.Controls.Add(lbl)


        prefix_spaces = ' ' * 18

        for index, line in enumerate(lines):
            if line.strip() != '':
                lines[index] = f'  [ {dt.hour:02d}:{dt.minute:02d}:{dt.second:02d} ]    ' + line
                break


        lines = lines[:index+1] + [prefix_spaces + i for i in lines[index+1:]]


        log = [lbl.Text for lbl in self.panel.Controls]

        if len(log) >= len(lines) and log[-len(lines):] == lines:
            return
        

        for log_string in lines:

            if self.panel.Controls.Count == 0 or log_string != self.panel.Controls[self.panel.Controls.Count - 1].Text: # no mostrar mensajes exactos consecutivos
                lbl = self.mk_log_lbl(log_string, color)
                self.panel.Controls.Add(lbl)



        n_del = 0

        while self.panel.Controls.Count >= 100:

            if n_del == 0: # 1st iter
                self.panel.SuspendLayout()

            self.panel.Controls.Remove(self.panel.Controls[0])
            n_del += 1

        if n_del > 0:

            for lbl in self.panel.Controls:
                lbl.Location = Point(lbl.Location.X, lbl.Location.Y - lbl.Size.Height*n_del)

            self.panel.ResumeLayout()


        self.panel.AutoScrollPosition = Point(0, self.panel.VerticalScroll.Maximum)
        #self.panel.Refresh()

        self.last_dt = dt




        








def create_main_form(output_script_path):

    cwd = os.path.split(sys.argv[0])[0] or '.'
    os.chdir(cwd)

    main_form = MainForm(output_script_path)

    thread = Thread(ThreadStart(lambda: (
        Application.EnableVisualStyles(),
        Application.Run(main_form)
    )))

    thread.SetApartmentState(ApartmentState.STA)
    thread.Start()
    #thread.Join()

    return main_form